// DOM Elements
const socialButtons = document.querySelectorAll('.social-btn');
const sections = document.querySelectorAll('section');

// Add fade-in animation to sections on load
document.addEventListener('DOMContentLoaded', () => {
    // Add fade-in effect to all sections
    sections.forEach((section, index) => {
        setTimeout(() => {
            section.classList.add('fade-in');
        }, index * 100);
    });
    
    // Add hover effects for social buttons
    socialButtons.forEach(button => {
        button.addEventListener('mouseenter', () => {
            button.style.transform = 'translateY(-2px) scale(1.05)';
        });
        
        button.addEventListener('mouseleave', () => {
            button.style.transform = 'translateY(0) scale(1)';
        });
    });
});

// Smooth scrolling for any internal links
document.querySelectorAll('a[href^="#"]').forEach(anchor => {
    anchor.addEventListener('click', function (e) {
        e.preventDefault();
        const target = document.querySelector(this.getAttribute('href'));
        
        if (target) {
            target.scrollIntoView({
                behavior: 'smooth',
                block: 'start'
            });
        }
    });
});

// Intersection Observer for animations
const observerOptions = {
    threshold: 0.1,
    rootMargin: '0px 0px -50px 0px'
};

const observer = new IntersectionObserver((entries) => {
    entries.forEach(entry => {
        if (entry.isIntersecting) {
            entry.target.style.opacity = '1';
            entry.target.style.transform = 'translateY(0)';
        }
    });
}, observerOptions);

// Observe FAQ items for animation
const faqItems = document.querySelectorAll('.faq-item');
faqItems.forEach(item => {
    item.style.opacity = '0';
    item.style.transform = 'translateY(20px)';
    observer.observe(item);
});

// Add click tracking for social buttons (optional analytics)
socialButtons.forEach(button => {
    button.addEventListener('click', function() {
        const platform = this.classList[1]; // gets the social platform class
        console.log(`User clicked on ${platform} button`);
        // Here you could add analytics tracking if needed
    });
});

// Simple notification system for link interactions
function showNotification(message, type = 'info') {
    // Remove existing notifications
    const existingNotification = document.querySelector('.notification');
    if (existingNotification) {
        existingNotification.remove();
    }
    
    // Create notification element
    const notification = document.createElement('div');
    notification.className = `notification notification-${type}`;
    notification.innerHTML = `
        <div class="notification-content">
            <span class="notification-message">${message}</span>
            <button class="notification-close">&times;</button>
        </div>
    `;
    
    // Add styles
    notification.style.cssText = `
        position: fixed;
        top: 20px;
        right: 20px;
        z-index: 10000;
        max-width: 350px;
        padding: 15px;
        border-radius: 8px;
        color: white;
        transform: translateX(100%);
        transition: transform 0.3s ease;
        box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
        ${type === 'success' ? 'background: #28a745;' : ''}
        ${type === 'info' ? 'background: #007bff;' : ''}
    `;
    
    notification.querySelector('.notification-content').style.cssText = `
        display: flex;
        justify-content: space-between;
        align-items: center;
        gap: 10px;
    `;
    
    notification.querySelector('.notification-close').style.cssText = `
        background: none;
        border: none;
        color: white;
        font-size: 1.5rem;
        cursor: pointer;
        padding: 0;
        line-height: 1;
    `;
    
    // Add to page
    document.body.appendChild(notification);
    
    // Show notification
    setTimeout(() => {
        notification.style.transform = 'translateX(0)';
    }, 100);
    
    // Auto hide after 3 seconds
    setTimeout(() => {
        hideNotification(notification);
    }, 3000);
    
    // Close button functionality
    notification.querySelector('.notification-close').addEventListener('click', () => {
        hideNotification(notification);
    });
}

function hideNotification(notification) {
    notification.style.transform = 'translateX(100%)';
    setTimeout(() => {
        if (notification.parentNode) {
            notification.parentNode.removeChild(notification);
        }
    }, 300);
}

// Handle external link clicks with notifications
document.addEventListener('click', (e) => {
    const link = e.target.closest('a[target="_blank"]');
    if (link && !link.classList.contains('social-btn')) {
        const platform = link.href.includes('telegram') ? 'Telegram' : 
                        link.href.includes('whatsapp') ? 'WhatsApp' :
                        link.href.includes('viber') ? 'Viber' :
                        link.href.includes('signal') ? 'Signal' : 'external site';
        
        showNotification(`Opening ${platform}...`, 'info');
    }
});

// Keyboard accessibility
document.addEventListener('keydown', (e) => {
    // Handle Enter key on social buttons
    if (e.key === 'Enter' && e.target.classList.contains('social-btn')) {
        e.target.click();
    }
    
    // ESC key closes notifications
    if (e.key === 'Escape') {
        const notification = document.querySelector('.notification');
        if (notification) {
            hideNotification(notification);
        }
    }
});


// Simple error handling
window.addEventListener('error', (e) => {
    console.warn('An error occurred:', e.error);
    // Could show user-friendly error message here if needed
});

// Add some interactivity to the FAQ items
const faqItemsInteractive = document.querySelectorAll('.faq-item');
faqItemsInteractive.forEach(item => {
    item.addEventListener('mouseenter', () => {
        item.style.transform = 'translateY(-2px)';
        item.style.boxShadow = '0 4px 15px rgba(0, 0, 0, 0.1)';
    });
    
    item.addEventListener('mouseleave', () => {
        item.style.transform = 'translateY(0)';
        item.style.boxShadow = 'none';
    });
});

// Console welcome message
console.log('🌿 Welcome to KingBerliner community site!');
console.log('Join our Telegram group for the best cannabis community in Berlin.');